/* 
 *  InteractionTimerMap.java
 * 
 *  Copyright 2016 Avaya Inc. All Rights Reserved.
 * 
 *  Usage of this source is bound to the terms described
 *  in AvayaLicenseSDK.rtf.
 * 
 *  Avaya - Confidential & Proprietary. Use pursuant to your signed agreement
 *  or Avaya Policy
 * 
 */
package com.avaya.ccs.javafxrefclient;

import com.avaya.ccs.api.ContactNumberI;
import java.util.HashMap;

public class InteractionTimerMap {

    private static final HashMap<String, InteractionTimer> TIMERS = new HashMap<>();
    private static final Logger LOG = Logger.getLogger(InteractionTimerMap.class);

    public InteractionTimerMap() {

    }

    /**
     * Stop any timer found with the matching interactionID
     *
     * @param interactionID
     * @return true if a timer was found stopped and removed, false if no timer
     * was found.
     */
    public boolean stop(String interactionID) {
        String methodName = "stop() ";
        synchronized (this) {
            InteractionTimer t = TIMERS.get(interactionID);
            if (t == null) {
                LOG.trace(methodName + " no timer to stop with interactonID:" + interactionID);
                return false;
            } else {
                LOG.debug(methodName + " stopping and removing timer with interactonID:" + interactionID);
                t.stopTimer();
                TIMERS.remove(interactionID);
                return true;
            }
        }
    }

    /**
     * Start preview timer, stopping and removing any existing timer first.
     *
     * @param timerLength
     * @param interactionID
     * @param todial
     * @param gui
     */
    public void startPreview(int timerLength, String interactionID, ContactNumberI todial, InteractionDataGUI gui) {
        String methodName = "startPreview() ";
        synchronized (this) {
            InteractionTimer t = TIMERS.get(interactionID);
            if ((t == null) || !(t instanceof PreviewTimer)) {
                LOG.debug(methodName + " starting timer with interactonID:" + interactionID);
                stop(interactionID);
                PreviewTimer pt = new PreviewTimer(timerLength, interactionID, todial, gui);
                TIMERS.put(interactionID, pt);
            } else {
                LOG.debug(methodName + " timer already running with interactonID:" + interactionID);
            }
        }
    }

    /**
     * Start wrapup timer, stopping and removing any existing timer first
     *
     * @param timerLength
     * @param interactionID
     * @param gui
     */
    public void startWrapUp(int timerLength, String interactionID, InteractionDataGUI gui) {
        String methodName = "startWrapUp() ";
        synchronized (this) {
            InteractionTimer t = TIMERS.get(interactionID);
            if ((t == null) || !(t instanceof WrapUpTimer)) {
                LOG.debug(methodName + " starting timer with interactonID:" + interactionID);
                stop(interactionID);
                WrapUpTimer wt = new WrapUpTimer(timerLength, interactionID, gui);
                TIMERS.put(interactionID, wt);
            } else {
                LOG.debug(methodName + " timer already running with interactonID:" + interactionID);
            }
        }
    }

    /**
     * Restart the wrapup timer
     *
     * @param interactionID
     * @return true if a wrapup timer was found and successfully restarted,
     * false otherwise
     */
    public boolean restartWrapUP(String interactionID) {
        String methodName = "restartWrapUP() ";
        synchronized (this) {
            InteractionTimer t = TIMERS.get(interactionID);
            if (t == null) {
                LOG.debug(methodName + " no timer to restart with interactonID:" + interactionID);
                return false;
            } else if (t instanceof WrapUpTimer) {
                LOG.debug(methodName + " restarting wrapup timer with interactonID:" + interactionID);
                return t.restartTimer();
            } else {
                LOG.debug(methodName + " no wrapup timer to restart with interactonID:" + interactionID);
                return false;
            }
        }
    }
    
    /**
     * Extend the wrapup timer, adding the origianl timer value to the current time remaining
     *
     * @param interactionID
     * @return true if a wrapup timer was found and successfully extended,
     * false otherwise
     */
    public boolean extendWrapUP(String interactionID) {
        String methodName = "extendWrapUP() ";
        synchronized (this) {
            InteractionTimer t = TIMERS.get(interactionID);
            if (t == null) {
                LOG.debug(methodName + " no timer to extend with interactonID:" + interactionID);
                return false;
            } else if (t instanceof WrapUpTimer) {
                LOG.debug(methodName + " extending wrapup timer with interactonID:" + interactionID);
                return t.extendTimer();
            } else {
                LOG.debug(methodName + " no wrapup timer to extend with interactonID:" + interactionID);
                return false;
            }
        }
    }
}
